<?php
/*-----------------------------------------------------------------------------------*/
/* General Front-end Functions
/*-----------------------------------------------------------------------------------*/

/**
 * Initiate Front-end
 *
 * @since 2.1.0
 */

if( ! function_exists( 'radium_frontend_init' ) ) {
	function radium_frontend_init() {

		global $post;
		
		$framework = radium_framework();
		
		/*------------------------------------------------------*/
		/* Primary Post ID
		/*------------------------------------------------------*/
		
		$primary = null;
		if( is_object( $post ) )
			$primary = $post->ID;
		
		/*------------------------------------------------------*/
		/* Fake Conditional
		/*------------------------------------------------------*/
		
		// This can be used to replace any conditional statements that 
		// come after any usages of query_posts
		
		$fake_conditional = radium_get_fake_conditional();
		
		/*------------------------------------------------------*/
		/* Builder (ID of custom layout or false)
		/*------------------------------------------------------*/
			
		$builder = false;
		
		// Custom Layout on static page
		if( is_page_template( 'page-builder.php' ) ) {
			$layout_id = get_post_meta( $post->ID, '_radium_custom_layout', true );
			if( $layout_id ) {
				$builder = $layout_id;
				$layout_post_id = radium_post_id_by_name( $layout_id, 'radium_layout' );
			} else {
				$builder = 'error';
			}
		}
		
		// Custom Layout on homepage
		if( is_home() ) {
			$homepage_content = radium_get_option( 'homepage_content', null, 'post_list' );
			if( $homepage_content == 'custom_layout' ) {
				$layout_id = radium_get_option( 'homepage_custom_layout' );
				if( $layout_id ) {
					$builder = $layout_id;
					$layout_post_id = radium_post_id_by_name( $layout_id, 'radium_layout' );
				} else {
					$builder = 'error';
				}
			}
		}
		
		/*------------------------------------------------------*/
		/* Featured Area
		/*------------------------------------------------------*/		
		$featured = array();
		
		if( $builder ) {
			$layout_post_id = radium_post_id_by_name( $builder, 'radium_layout' );
			$elements = get_post_meta( $layout_post_id, 'elements', true );
			
			// Set classes for featured area
			if( is_array( $elements ) && isset( $elements['featured'] ) && ! empty( $elements['featured'] ) ){
				$featured[] = 'has_builder';
				foreach( $elements['featured'] as $element )
 					$featured[] = $element['type'];
			}
  					
		}
 
		/*------------------------------------------------------*/
		/* Finalize Frontend Configuration
		/*------------------------------------------------------*/
		
    	$config = array(
    		'id'				=> $primary,			// global $post->ID that can be accessed anywhere
    		'fake_conditional'	=> $fake_conditional,	// Fake conditional tag
    		'builder'			=> $builder,			// ID of current custom layout if not false
    		'featured'			=> $featured,			// Classes for featured areas, if empty featured area won't show
    	);
    	$config = apply_filters( 'radium_frontend_config', $config );
    	
    	return $config;
	}
}

/**
 * Add framework css classes to body_class() 
 *
 * @since 2.0.2
 *
 * @param array $classes Current WordPress body classes
 * @return array $classes Modified body classes
 */

if( ! function_exists( 'radium_builder_body_class' ) ) {
	function radium_builder_body_class( $classes ) {
		
		// Featured Area
		if( radium_builder_config( 'featured' ) )
			$classes[] = 'show-featured-area';
		else
			$classes[] = 'hide-featured-area';

		// Custom Layout
		$custom_layout = radium_builder_config( 'builder' );
		if( $custom_layout )
			$classes[] = 'custom-layout-'.$custom_layout;
		
		return $classes;
	}
}

/**
 * Set fake conditional.
 *
 * Because query_posts alters the current global $wp_query 
 * conditional, this function is called before query_posts 
 * and assigns a variable to act as a fake conditional if 
 * needed after query_posts.
 *
 * @since 2.1.0
 *
 * @return string $fake_condtional HTML to output thumbnail
 */

if( ! function_exists( 'radium_get_fake_conditional' ) ) {
	function radium_get_fake_conditional() {
		$fake_conditional = '';
		if( is_home() )
			$fake_conditional = 'home';
		else if( is_page_template( 'page-builder.php' ) )
			$fake_conditional = 'page-builder.php';
		else if( is_single() )
			$fake_conditional = 'single';
		else if( is_search() )
			$fake_conditional = 'search';
		else if ( is_archive() )
			$fake_conditional = 'archive';
		return $fake_conditional;
	}	
}

/**
 * This function is used from within the theme's template 
 * files to return the values setup in the previous init function.
 *
 * @since 2.1.0
 * 
 * @param $key string $key to retrieve from $config
 * @return $value mixed value from $config
 */

function radium_builder_config( $key, $seconday = null ) {

	$config = radium_frontend_init();
	
	$value = null;
	if( $seconday ) {
		if( isset( $config[$key][$seconday] ) )
			$value = $config[$key][$seconday];	
	} else {	
		if( isset( $config[$key] ) )
			$value = $config[$key];
	}
	return $value;
}


/**
 * Display custom layout within page-builder.php 
 * page template.
 *
 * @since 2.1.0
 *
 * @param string $layout Post slug for layout
 * @param string $location Location of elements, featured or primary
 */
 
if( ! function_exists( 'radium_builder_elements' ) ) {
	function radium_builder_elements( $layout, $location ) {
		
		$counter = 1;
		$primary_query = false;
		$layout_id = radium_post_id_by_name( $layout, 'radium_layout' );
		
		//Die if no Layout ID is found
		if( ! $layout_id ) {
			__( 'No Layout found for this page.', 'radium' );
			return;
		}

		$elements = get_post_meta( $layout_id, 'elements', true );
		
		if( is_array( $elements ) && isset( $elements[$location] ) && ! empty( $elements[$location] ) ) {
			
			$elements = $elements[$location];
			$num_elements = count($elements);
			
		} else {
			return;
		}

		// Loop through elements
		foreach( $elements as $id => $element ) {
			
			//translate width data into a usable format
			$element_width = isset($element['width']) ? $element['width'] : null;
			
			switch ( $element_width ) {
				case 'element1-1':
					$element_width = 'twelve columns';
					break;
				case 'element1-2':
					$element_width = 'six columns';
					break;
				case 'element1-3':
					$element_width = 'four columns';
					break;
				case 'element1-4':
					$element_width = 'three columns';
					break;
				case 'element2-3':
					$element_width = 'eight columns';
					break;	
				case 'element3-4':
					$element_width = 'nine columns';
					break;	
				default:
					$element_width = 'twelve columns';
					break;				
			}
			
			// classes
			$classes = 'element '.$location.'-element-'.$counter.' element-'.$element['type'];
			if( $counter == 1 )
				$classes .= ' first-element';
			if( $num_elements == $counter )
				$classes .= ' last-element';
			if( $element['type'] == 'slider' ) {
				$slider_id = radium_post_id_by_name( $element['options']['slider_id'], 'slider' );
				$type = radium_get_custom_field('_radium_slider_type', $slider_id );
				$classes .= ' element-slider-'.$type;
			}
			
			// Start output
			echo '<div class="'.$classes . ' ' . $element_width .'">';
				
			//element blocks are added here
			do_action( 'radium_builder_'.$element['type'], $id, $element['options'], $location );
			
			echo '<div class="clear"></div>';
			echo '</div><!-- .element  -->';
			
			$counter++;
		}
				
	}
}

include('elements/blog-element.php');
include('elements/cta-element.php');
include('elements/carousel-element.php');
include('elements/content-element.php');
include('elements/divider-element.php');
include('elements/featured-area-element.php');
include('elements/headline-element.php');
include('elements/map-element.php');
include('elements/slider-element.php');
include('elements/slogan-element.php');
include('elements/widget-element.php');